import {
  StateNode,
  Vec
} from "@tldraw/editor";
import { getArrowBindings } from "../../../shapes/arrow/shared.mjs";
import {
  NOTE_CENTER_OFFSET,
  getNoteAdjacentPositions,
  getNoteShapeForAdjacentPosition
} from "../../../shapes/note/noteHelpers.mjs";
import { startEditingShapeWithLabel } from "../selectHelpers.mjs";
class PointingHandle extends StateNode {
  static id = "pointing_handle";
  info = {};
  onEnter = (info) => {
    this.info = info;
    const { shape } = info;
    if (this.editor.isShapeOfType(shape, "arrow")) {
      const initialBinding = getArrowBindings(this.editor, shape)[info.handle.id];
      if (initialBinding) {
        this.editor.setHintingShapes([initialBinding.toId]);
      }
    }
    this.editor.setCursor({ type: "grabbing", rotation: 0 });
  };
  onExit = () => {
    this.editor.setHintingShapes([]);
    this.editor.setCursor({ type: "default", rotation: 0 });
  };
  onPointerUp = () => {
    const { shape, handle } = this.info;
    if (this.editor.isShapeOfType(shape, "note")) {
      const { editor } = this;
      const nextNote = getNoteForPit(editor, shape, handle, false);
      if (nextNote) {
        startEditingShapeWithLabel(
          editor,
          nextNote,
          true
          /* selectAll */
        );
        return;
      }
    }
    this.parent.transition("idle", this.info);
  };
  onPointerMove = () => {
    const { editor } = this;
    if (editor.inputs.isDragging) {
      this.startDraggingHandle();
    }
  };
  onLongPress = () => {
    this.startDraggingHandle();
  };
  startDraggingHandle() {
    const { editor } = this;
    if (editor.getInstanceState().isReadonly) return;
    const { shape, handle } = this.info;
    if (editor.isShapeOfType(shape, "note")) {
      const nextNote = getNoteForPit(editor, shape, handle, true);
      if (nextNote) {
        const centeredOnPointer = editor.getPointInParentSpace(nextNote, editor.inputs.originPagePoint).sub(Vec.Rot(NOTE_CENTER_OFFSET.clone().mul(shape.props.scale), nextNote.rotation));
        editor.updateShape({ ...nextNote, x: centeredOnPointer.x, y: centeredOnPointer.y });
        editor.setHoveredShape(nextNote.id).select(nextNote.id).setCurrentTool("select.translating", {
          ...this.info,
          target: "shape",
          shape: editor.getShape(nextNote),
          onInteractionEnd: "note",
          isCreating: true,
          onCreate: () => {
            startEditingShapeWithLabel(
              editor,
              nextNote,
              true
              /* selectAll */
            );
          }
        });
        return;
      }
    }
    this.parent.transition("dragging_handle", this.info);
  }
  onCancel = () => {
    this.cancel();
  };
  onComplete = () => {
    this.cancel();
  };
  onInterrupt = () => {
    this.cancel();
  };
  cancel() {
    this.parent.transition("idle");
  }
}
function getNoteForPit(editor, shape, handle, forceNew) {
  const pageTransform = editor.getShapePageTransform(shape.id);
  const pagePoint = pageTransform.point();
  const pageRotation = pageTransform.rotation();
  const pits = getNoteAdjacentPositions(
    editor,
    pagePoint,
    pageRotation,
    shape.props.growY,
    0,
    shape.props.scale
  );
  const pit = pits[handle.index];
  if (pit) {
    return getNoteShapeForAdjacentPosition(editor, shape, pit, pageRotation, forceNew);
  }
}
export {
  PointingHandle
};
//# sourceMappingURL=PointingHandle.mjs.map
